<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AlertController extends Controller
{
    /**
     * Page principale des alertes (liste des réglages + dernières alertes).
     * GET /alerts?ref=ruche-001
     */
    public function index(Request $request)
    {
        // Liste des ruches (par leur référence)
        $ruches = DB::table('ruches')->orderBy('reference')->pluck('reference')->toArray();

        // Ruche active (par défaut: première, sinon 'ruche-001')
        $ref = $request->query('ref') ?: ($ruches[0] ?? 'ruche-001');

        $ruche = DB::table('ruches')->where('reference', $ref)->first();
        if (!$ruche) {
            abort(404, "Ruche inconnue: {$ref}");
        }

        // Réglages d'alerte pour cette ruche
        $settings = DB::table('alert_settings as s')
            ->where('s.ruche_id', $ruche->id)
            ->orderBy('s.type')
            ->get();

        // Dernières alertes — PAS de join sur capteurs (pas de capteur_id dans alerts)
        $latestAlerts = DB::table('alerts as a')
            ->where('a.ruche_id', $ruche->id)
            ->orderByDesc('a.created_at')
            ->limit(50)
            ->get();

        return view('alerts', [
            'ref'          => $ref,
            'ruches'       => $ruches,
            'settings'     => $settings,
            'latestAlerts' => $latestAlerts,
        ]);
    }

    /**
     * Crée les réglages par défaut si manquants (pour une ruche donnée ou pour toutes).
     * POST /alerts/bootstrap  (body: ref=ruche-001 optionnel, types=temperature,humidity optionnel)
     */
    public function bootstrap(Request $request)
    {
        $ref   = $request->input('ref');    // si null -> toutes les ruches
        $types = $request->input('types');  // "temperature,humidity" (optionnel)

        // Valeurs par défaut (modifiable)
        $defaults = [
            'temperature' => ['threshold_min' => 25,   'threshold_max' => 35,   'hysteresis' => 0.5],
            'humidity'    => ['threshold_min' => 30,   'threshold_max' => 85,   'hysteresis' => 2],
            'weight'      => ['threshold_min' => null, 'threshold_max' => null, 'hysteresis' => 0.5],
            'battery'     => ['threshold_min' => 3.3,  'threshold_max' => 4.3,  'hysteresis' => 0.05],
        ];

        if ($types) {
            $typesArr = array_filter(array_map('trim', explode(',', $types)));
            $defaults = array_intersect_key($defaults, array_flip($typesArr));
        }

        $q = DB::table('ruches');
        if ($ref) $q->where('reference', $ref);
        $ruches = $q->get();

        foreach ($ruches as $ruche) {
            foreach ($defaults as $type => $cfg) {
                $exists = DB::table('alert_settings')
                    ->where(['ruche_id' => $ruche->id, 'type' => $type])
                    ->exists();

                if (!$exists) {
                    DB::table('alert_settings')->insert([
                        'ruche_id'      => $ruche->id,
                        'type'          => $type,
                        'threshold_min' => $cfg['threshold_min'],
                        'threshold_max' => $cfg['threshold_max'],
                        'hysteresis'    => $cfg['hysteresis'],
                        'enabled'       => 1,
                        'created_at'    => now(),
                        'updated_at'    => now(),
                    ]);
                }
            }
        }

        return back()->with('status', 'Réglages d’alerte initialisés.');
    }

    /**
     * Met à jour une ligne de réglage (seuils/hystérésis/activé) avec normalisation.
     * POST /alerts/{id}/update
     */
    public function update(Request $request, $id)
    {
        // Lecture et normalisation
        $min = $request->input('threshold_min');
        $max = $request->input('threshold_max');
        $hys = $request->input('hysteresis');
        $ena = $request->boolean('enabled'); // cast bool

        $min = ($min === '' || $min === null) ? null : $min;
        $max = ($max === '' || $max === null) ? null : $max;

        // Validation
        $request->validate([
            'hysteresis' => ['required', 'numeric', 'min:0'],
        ]);

        if (!is_null($min) && !is_numeric($min)) {
            return back()->withErrors(['threshold_min' => 'Seuil min doit être numérique ou vide.'])->withInput();
        }
        if (!is_null($max) && !is_numeric($max)) {
            return back()->withErrors(['threshold_max' => 'Seuil max doit être numérique ou vide.'])->withInput();
        }
        if (!is_null($min) && !is_null($max) && (float)$min > (float)$max) {
            return back()->withErrors(['threshold_min' => 'Seuil min doit être ≤ seuil max.'])->withInput();
        }

        // Enregistrement
        DB::table('alert_settings')->where('id', $id)->update([
            'threshold_min' => is_null($min) ? null : (float)$min,
            'threshold_max' => is_null($max) ? null : (float)$max,
            'hysteresis'    => (float)$hys,
            'enabled'       => $ena ? 1 : 0,
            'updated_at'    => now(),
        ]);

        return back()->with('status', 'Réglage d’alerte mis à jour.');
    }

    /**
     * Supprime une ligne de réglage.
     * DELETE /alerts/{id}
     */
    public function destroy($id)
    {
        DB::table('alert_settings')->where('id', $id)->delete();
        return back()->with('status', 'Réglage supprimé.');
    }
}
