<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Carbon\Carbon;

class DashboardController extends Controller
{
    /**
     * Affiche le dashboard d'une ruche.
     *
     * Comportement :
     * - Résout la ruche active (param URL {ref}, ou ?ref=..., sinon 1ère ruche, sinon 'ruche-001').
     * - Valide que la ruche existe en BDD, sinon redirige vers la 1ère disponible.
     * - Passe la liste des ruches au Blade.
     * - Passe aussi une fenêtre temporelle initiale (24h glissantes).
     * - (Optionnel) Pré-hydrate le Blade avec quelques infos (compteurs, existe_mesures).
     *
     * NB : Les séries temporelles NE sont PAS chargées ici : le Blade appelle l'API
     * `/api/ruche/{ref}/series?from=&to=` côté navigateur (fetch), ce qui garde le contrôleur simple.
     */
    public function index(Request $request, ?string $ref = null)
    {
        // --- Liste des ruches pour le sélecteur ---
        $ruches = DB::table('ruches')
            ->select('reference')
            ->orderBy('reference')
            ->pluck('reference');

        // Aucune ruche ? On tolère mais on affiche quand même la page.
        $firstRef = $ruches->first();

        // --- Détermination de la ref active ---
        $ref = $ref
            ?? $request->query('ref')
            ?? ($firstRef ?: 'ruche-001');

        // Normalisation basique (optionnelle)
        $ref = trim($ref);

        // --- Validation : la ruche doit exister si on en a au moins une en BDD ---
        if ($firstRef && !$ruches->contains($ref)) {
            // ref inconnue -> redirection douce vers la 1ère ruche existante
            return redirect()->route('dashboard.index', ['ref' => $firstRef]);
        }

        // --- Fenêtre temporelle par défaut : 24h glissantes ---
        $to   = Carbon::now();                 // timezone app()
        $from = (clone $to)->subHours(24);

        // Permet d’override via querystring ?from=...&to=...
        // On ne force pas ici, on laisse le JS du Blade décider. Mais si tu veux,
        // décommente :
        /*
        if ($request->filled('from')) {
            $from = Carbon::parse($request->query('from'), config('app.timezone'));
        }
        if ($request->filled('to')) {
            $to = Carbon::parse($request->query('to'), config('app.timezone'));
        }
        */

        // --- (Optionnel) Méta rapides pour le Blade (pas indispensable) ---
        // Exemple : vérifier qu’il existe des mesures sur cette ruche
        $hasMesures = false;
        try {
            $hasMesures = DB::table('mesures')
                ->join('capteurs', 'capteurs.id', '=', 'mesures.capteur_id')
                ->join('ruches', 'ruches.id', '=', 'capteurs.ruche_id')
                ->where('ruches.reference', $ref)
                ->exists();
        } catch (\Throwable $e) {
            // On n’échoue pas la page pour ça, on laisse false
        }

        // --- Construction du view-model ---
        return view('dashboard', [
            'ref'         => $ref,
            'ruches'      => $ruches,
            // Infos utiles si tu veux les exploiter côté Blade (facultatif)
            'initialFrom' => $from->toIso8601String(),
            'initialTo'   => $to->toIso8601String(),
            'hasMesures'  => $hasMesures,
        ]);
    }
}
