<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Symfony\Component\HttpFoundation\StreamedResponse;
use Carbon\Carbon;

class ExportController extends Controller
{
    /**
     * GET /export/{ref}.csv?from=ISO&to=ISO
     * Exporte les mesures d'une ruche en CSV (dernières 24h par défaut).
     */
    public function exportCsv(string $ref, Request $request)
    {
        // 1) Ruche
        $ruche = DB::table('ruches')->where('reference', $ref)->first();
        if (!$ruche) {
            abort(404, "Ruche inconnue: {$ref}");
        }

        // 2) Fenêtre temporelle
        $toParam   = $request->query('to');
        $fromParam = $request->query('from');

        try {
            $to = $toParam ? Carbon::parse($toParam) : Carbon::now();
        } catch (\Throwable $e) {
            $to = Carbon::now();
        }
        try {
            $from = $fromParam ? Carbon::parse($fromParam) : $to->copy()->subDay();
        } catch (\Throwable $e) {
            $from = $to->copy()->subDay();
        }

        // 3) Requête : utilise uniquement les colonnes qui existent (value_numeric, valeur)
        $rows = DB::table('mesures as m')
            ->join('capteurs as c', 'c.id', '=', 'm.capteur_id')
            ->where('c.ruche_id', $ruche->id)
            ->whereBetween('m.measured_at', [$from->toDateTimeString(), $to->toDateTimeString()])
            ->orderBy('m.measured_at', 'asc')
            ->select([
                'm.measured_at',
                DB::raw('c.type as type'),
                DB::raw('COALESCE(m.value_numeric, m.valeur) as value'),
                'm.capteur_id',
            ])
            ->get();

        // 4) Stream CSV
        $filename = sprintf(
            'ruche_%s_%s_%s.csv',
            $ref,
            $from->format('Ymd_His'),
            $to->format('Ymd_His')
        );

        $headers = [
            'Content-Type'        => 'text/csv; charset=UTF-8',
            'Content-Disposition' => 'attachment; filename="'.$filename.'"',
        ];

        $callback = function () use ($rows) {
            $out = fopen('php://output', 'w');
            // BOM UTF-8 (compatible Excel)
            fprintf($out, chr(0xEF).chr(0xBB).chr(0xBF));
            // En-têtes
            fputcsv($out, ['measured_at', 'type', 'value', 'capteur_id']);
            foreach ($rows as $r) {
                fputcsv($out, [
                    $r->measured_at,
                    $r->type,
                    is_numeric($r->value) ? (float)$r->value : $r->value,
                    $r->capteur_id,
                ]);
            }
            fclose($out);
        };

        return new StreamedResponse($callback, 200, $headers);
    }
}
