<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Models\Ruche;
use App\Models\Capteur;
use App\Models\Mesure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class RucheController extends Controller
{
    public function index()
    {
        $ruches = Ruche::with('rucher')->orderBy('reference')->paginate(15);
        return view('ruches.index', compact('ruches'));
    }

    public function show(Ruche $ruche)
    {
        $ruche->load(['rucher', 'capteurs', 'alertes' => function($q) {
            $q->latest()->limit(10);
        }]);

        $latest = Mesure::select('capteur_id', DB::raw('MAX(mesure_at) as last_at'))
            ->whereIn('capteur_id', $ruche->capteurs->pluck('id'))
            ->groupBy('capteur_id')
            ->pluck('last_at', 'capteur_id');

        return view('ruches.show', compact('ruche', 'latest'));
    }

    public function timeseries(Ruche $ruche, Request $request)
    {
        $type  = $request->query('type');
        $limit = (int) $request->query('limit', 100);
        $limit = max(10, min($limit, 1000));

        $query = Mesure::query()
            ->select('mesures.id','mesures.capteur_id','mesures.valeur_numeric','mesures.valeur_text','mesures.mesure_at')
            ->join('capteurs','capteurs.id','=','mesures.capteur_id')
            ->where('capteurs.ruche_id', $ruche->id)
            ->orderByDesc('mesures.mesure_at')
            ->limit($limit);

        if ($type) {
            $query->where('capteurs.type', $type);
        }

        $rows = $query->get()->reverse()->values()->map(function ($m) {
            return [
                't'   => optional($m->mesure_at)->toIso8601String(),
                'val' => $m->valeur_numeric ?? $m->valeur_text,
            ];
        });

        return response()->json([
            'reference' => $ruche->reference,
            'type'      => $type,
            'count'     => $rows->count(),
            'data'      => $rows,
        ]);
    }
}
