<?php

namespace App\Services;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class AlertEngine
{
    /**
     * Scanne les mesures et génère des alertes selon alert_settings (hystérésis).
     * @param string|null $ref   Référence ruche (ex: 'ruche-001'). Si null => toutes.
     * @param int         $hours Fenêtre en heures à scanner (ex: 24).
     * @return array { scanned:int, created:int, auto_acked:int, skipped:int }
     */
    public static function scan(?string $ref = null, int $hours = 24): array
    {
        $created   = 0;
        $autoAcked = 0;
        $scanned   = 0;
        $skipped   = 0;

        // Détecte la colonne valeur existante dans "mesures"
        $hasValueNum = Schema::hasColumn('mesures', 'value_numeric');
        $hasValue    = Schema::hasColumn('mesures', 'value');
        $hasValeur   = Schema::hasColumn('mesures', 'valeur');
        $valueExpr   = $hasValueNum ? 'm.value_numeric'
                    : ($hasValue ? 'm.value'
                    : ($hasValeur ? 'm.valeur' : 'NULL'));

        // Sélection des ruches
        $rq = DB::table('ruches')->select('id','reference')->orderBy('id');
        if ($ref) $rq->where('reference', $ref);
        $ruches = $rq->get();

        foreach ($ruches as $ruche) {
            // Règles actives
            $settings = DB::table('alert_settings')
                ->where('ruche_id', $ruche->id)
                ->where('enabled', 1)
                ->get();

            if ($settings->isEmpty()) {
                // rien à scanner pour cette ruche
                $skipped++;
                continue;
            }

            $from = now()->subHours($hours);

            foreach ($settings as $s) {
                $type = trim((string)($s->type ?? ''));
                if ($type === '') { $skipped++; continue; }

                // Points de la fenêtre
                $points = DB::table('mesures as m')
                    ->join('capteurs as c','c.id','=','m.capteur_id')
                    ->where('c.ruche_id', $ruche->id)
                    ->where('c.type', $type)
                    ->where('m.measured_at','>=',$from)
                    ->orderBy('m.measured_at','asc')
                    ->select('m.measured_at', DB::raw("$valueExpr as v"))
                    ->get();

                if ($points->isEmpty()) { $skipped++; continue; }

                // Alerte ouverte actuelle pour ce type
                $lastOpen = DB::table('alerts')
                    ->where('ruche_id', $ruche->id)
                    ->where('type', $type)
                    ->whereNull('ack_at')
                    ->orderByDesc('measured_at')
                    ->first();

                $openLevel = $lastOpen->level ?? null; // 'low' | 'high' | null

                foreach ($points as $p) {
                    $scanned++;
                    $v = is_null($p->v) ? null : (float)$p->v;
                    if ($v === null) continue;

                    $tmin = is_null($s->threshold_min) ? null : (float)$s->threshold_min;
                    $tmax = is_null($s->threshold_max) ? null : (float)$s->threshold_max;
                    $hyst = (float)$s->hysteresis;

                    $breachHigh = !is_null($tmax) && ($v > $tmax);
                    $breachLow  = !is_null($tmin) && ($v < $tmin);

                    $highClear  = !is_null($tmax) && ($v < $tmax - $hyst);
                    $lowClear   = !is_null($tmin) && ($v > $tmin + $hyst);

                    // Résolution auto si on est revenu dans la zone OK (avec hysteresis)
                    if ($openLevel === 'high' && $highClear) {
                        DB::table('alerts')->where('id', $lastOpen->id)->update(['ack_at'=>now()]);
                        $openLevel = null;
                        $autoAcked++;
                    } elseif ($openLevel === 'low' && $lowClear) {
                        DB::table('alerts')->where('id', $lastOpen->id)->update(['ack_at'=>now()]);
                        $openLevel = null;
                        $autoAcked++;
                    }

                    // Création si aucun événement ouvert et dépassement
                    if ($openLevel === null) {
                        if ($breachHigh) {
                            DB::table('alerts')->insert([
                                'ruche_id'    => $ruche->id,
                                'type'        => $type,
                                'measured_at' => $p->measured_at,
                                'value'       => $v,
                                'level'       => 'high',
                                'ack_at'      => null,
                                'created_at'  => now(),
                                'updated_at'  => now(),
                            ]);
                            $created++;
                            $openLevel = 'high';
                            $lastOpen = (object)['id'=>DB::getPdo()->lastInsertId(),'level'=>'high'];
                        } elseif ($breachLow) {
                            DB::table('alerts')->insert([
                                'ruche_id'    => $ruche->id,
                                'type'        => $type,
                                'measured_at' => $p->measured_at,
                                'value'       => $v,
                                'level'       => 'low',
                                'ack_at'      => null,
                                'created_at'  => now(),
                                'updated_at'  => now(),
                            ]);
                            $created++;
                            $openLevel = 'low';
                            $lastOpen = (object)['id'=>DB::getPdo()->lastInsertId(),'level'=>'low'];
                        }
                    }
                } // points
            } // settings
        } // ruches

        return ['scanned'=>$scanned, 'created'=>$created, 'auto_acked'=>$autoAcked, 'skipped'=>$skipped];
    }
}
