<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Carbon\Carbon;

class DemoDataSeeder extends Seeder
{
    public function run(): void
    {
        $now = Carbon::now();

        // === 1) Apiculteur ===
        $apiculteurId = DB::table('apiculteurs')->where('email', 'demo@ruche.local')->value('id');
        if (!$apiculteurId) {
            $apiculteurId = DB::table('apiculteurs')->insertGetId([
                'nom'        => 'Demo',
                'prenom'     => 'Apiculteur',
                'telephone'  => '0000',
                'email'      => 'demo@ruche.local',
                'created_at' => $now,
                'updated_at' => $now,
            ]);
        }

        // === 2) Rucher ===
        $rucherId = DB::table('ruchers')->where('nom', 'Rucher Demo')->value('id');
        if (!$rucherId) {
            $rucherId = DB::table('ruchers')->insertGetId([
                'nom'          => 'Rucher Demo',
                'localisation' => 'Nouméa',
                'created_at'   => $now,
                'updated_at'   => $now,
            ]);
        }

        // === 3) Ruche ===
        $rucheId = DB::table('ruches')->where('reference', 'ruche-001')->value('id');
        if (!$rucheId) {
            $rucheId = DB::table('ruches')->insertGetId([
                'rucher_id'     => $rucherId,
                'apiculteur_id' => $apiculteurId,
                'reference'     => 'ruche-001',
                'description'   => 'Ruche de démonstration',
                'created_at'    => $now,
                'updated_at'    => $now,
            ]);
        }

        // === 4) Capteurs (temperature & humidity) ===
        $capteurTempId = DB::table('capteurs')
            ->where('ruche_id', $rucheId)
            ->where('type', 'temperature')
            ->value('id');

        if (!$capteurTempId) {
            $capteurTempId = DB::table('capteurs')->insertGetId([
                'ruche_id'   => $rucheId,
                'type'       => 'temperature',
                'modele'     => 'TMP36',
                'serial'     => (string) Str::uuid(),
                'created_at' => $now,
                'updated_at' => $now,
            ]);
        }

        $capteurHumId = DB::table('capteurs')
            ->where('ruche_id', $rucheId)
            ->where('type', 'humidity')
            ->value('id');

        if (!$capteurHumId) {
            $capteurHumId = DB::table('capteurs')->insertGetId([
                'ruche_id'   => $rucheId,
                'type'       => 'humidity',
                'modele'     => 'DHT22',
                'serial'     => (string) Str::uuid(),
                'created_at' => $now,
                'updated_at' => $now,
            ]);
        }

        // === 5) Mesures récentes (12h glissantes) ===
        // Une mesure toutes les 15 minutes => 48 points par capteur
        $points = 48;
        $start  = $now->copy()->subHours(12);

        // Helper idempotent : évite les doublons (même capteur_id + measured_at)
        $insertMesure = function (int $capteurId, Carbon $timestamp, float $value) use ($now) {
            $exists = DB::table('mesures')
                ->where('capteur_id', $capteurId)
                ->where('measured_at', $timestamp)
                ->exists();

            if (!$exists) {
                DB::table('mesures')->insert([
                    'capteur_id'    => $capteurId,
                    'measured_at'   => $timestamp,
                    'value_numeric' => $value,         // nouveau champ numérique
                    'valeur'        => $value,         // ancien champ (compatibilité)
                    'value_raw'     => null,           // si tu l'as ajouté ; sinon MySQL l'ignorera si absent
                    'created_at'    => $now,
                    'updated_at'    => $now,
                ]);
            }
        };

        for ($i = 0; $i < $points; $i++) {
            $t = $start->copy()->addMinutes($i * 15);

            // Température ~ 28°C +/- ~2
            $temp = 28 + sin($i / 6) * 1.5 + (mt_rand(-10, 10) / 10);
            // Humidité ~ 70% +/- ~5
            $hum  = 70 + cos($i / 7) * 3   + (mt_rand(-10, 10) / 10);

            $insertMesure($capteurTempId, $t, round($temp, 2));
            $insertMesure($capteurHumId,  $t, round($hum,  2));
        }
    }
}
