<?php

namespace App\Http\Controllers\Api;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Routing\Controller as BaseController;

class SeriesController extends BaseController
{
    /**
     * GET /api/ruche/{ref}/series?from=ISO&to=ISO&types=temperature,humidity&limit=5000
     * Retourne un JSON: { series: [{measured_at,type,value,capteur_id}, ...] }
     */
    public function index(Request $request, string $ref)
    {
        // Fenêtre temporelle (par défaut: 24h)
        $to   = $request->query('to');
        $from = $request->query('from');

        if (!$to)   { $to   = now()->toIso8601String(); }
        if (!$from) { $from = now()->subHours(24)->toIso8601String(); }

        // Filtre types optionnel
        $types = array_filter(array_map('trim', explode(',', (string)$request->query('types', ''))));
        $limit = (int)($request->query('limit', 5000));
        if ($limit <= 0 || $limit > 50000) { $limit = 5000; }

        // Compose la colonne de valeur disponible
        // (value_numeric prioritaire, sinon valeur; adapte si tu as "value")
        $valueExpr = 'COALESCE(m.value_numeric, m.valeur)';

        $q = DB::table('mesures as m')
            ->join('capteurs as c','c.id','=','m.capteur_id')
            ->join('ruches as r','r.id','=','c.ruche_id')
            ->where('r.reference', $ref)
            ->where('m.measured_at', '>=', $from)
            ->where('m.measured_at', '<=', $to)
            ->when(!empty($types), fn($qq) => $qq->whereIn('c.type', $types))
            ->orderBy('m.measured_at', 'asc')
            ->selectRaw('m.measured_at, c.type as type, '.$valueExpr.' as value, m.capteur_id');

        $rows = $q->limit($limit)->get();

        // Réponse standardisée pour le Blade (tolérant aux noms de clés)
        return response()->json([
            'ref'    => $ref,
            'from'   => $from,
            'to'     => $to,
            'count'  => $rows->count(),
            'series' => $rows,
        ]);
    }
}
