<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Carbon;

class RucheApiController extends Controller
{
    /**
     * Dernières valeurs (pour les tuiles du haut)
     * GET /api/ruches/{reference}/mesures/latest
     */
    public function latest(string $reference)
    {
        // Dernière température
        $lastTemp = DB::selectOne("
            SELECT m.valeur, m.measured_at
            FROM ruches r
            JOIN capteurs c ON c.ruche_id = r.id AND c.type = 'temperature'
            JOIN mesures  m ON m.capteur_id = c.id
            WHERE r.reference = ?
            ORDER BY m.measured_at DESC
            LIMIT 1
        ", [$reference]);

        // Dernière humidité
        $lastHum = DB::selectOne("
            SELECT m.valeur, m.measured_at
            FROM ruches r
            JOIN capteurs c ON c.ruche_id = r.id AND c.type = 'humidity'
            JOIN mesures  m ON m.capteur_id = c.id
            WHERE r.reference = ?
            ORDER BY m.measured_at DESC
            LIMIT 1
        ", [$reference]);

        return response()->json([
            'ruche'       => $reference,
            'temperature' => $lastTemp ? (float)$lastTemp->valeur : null,
            'temp_at'     => $lastTemp ? (string)$lastTemp->measured_at : null,
            'humidity'    => $lastHum ? (float)$lastHum->valeur : null,
            'hum_at'      => $lastHum ? (string)$lastHum->measured_at : null,
        ]);
    }

    /**
     * Historique formaté pour le graphe (Chart.js/Apex)
     * GET /api/ruches/{reference}/mesures/history?hours=24
     * Retourne:
     * {
     *   labels: ["2025-10-05 14:00:00", ...],
     *   series: { temperature: [28.5, null, ...], humidity: [null, 68, ...] }
     * }
     */
    public function history(Request $req, string $reference)
    {
        $hours = max(1, min(168, (int)$req->query('hours', 24)));
        $since = Carbon::now()->subHours($hours);

        // Récupération brute
        $rows = DB::select("
            SELECT m.measured_at, c.type, m.valeur
            FROM ruches r
            JOIN capteurs c ON c.ruche_id = r.id
            JOIN mesures  m ON m.capteur_id = c.id
            WHERE r.reference = ?
              AND m.measured_at >= ?
              AND c.type IN ('temperature','humidity')
            ORDER BY m.measured_at ASC
        ", [$reference, $since]);

        // Construction des labels + 2 séries parallèles
        $labels = [];
        $tData  = [];
        $hData  = [];

        foreach ($rows as $row) {
            $ts = Carbon::parse($row->measured_at)->format('Y-m-d H:i:s');

            // si nouveau timestamp, on pousse un point "vide" d'abord
            if (empty($labels) || end($labels) !== $ts) {
                $labels[] = $ts;
                $tData[]  = null;
                $hData[]  = null;
            }
            $idx = array_key_last($labels);

            if ($row->type === 'temperature') {
                $tData[$idx] = (float)$row->valeur;
            } elseif ($row->type === 'humidity') {
                $hData[$idx] = (float)$row->valeur;
            }
        }

        return response()->json([
            'ruche'  => $reference,
            'hours'  => $hours,
            'labels' => $labels,
            'series' => [
                'temperature' => $tData,
                'humidity'    => $hData,
            ],
        ]);
    }
}
