<?php

namespace App\Support;

final class TtnV1
{
    /**
     * Decode raw binary payload (bytes) for our v1 format:
     * stream of 16-bit segments: [type:2b][value:14b(Q1)]
     * 00 => temperature (signed Q1), 01 => humidity (unsigned Q1)
     * returns ['temperature'=>float?, 'humidity'=>float?] last occurrence wins
     */
    public static function decode(string $raw): array
    {
        $bits = self::bytesToBits($raw);
        $n = count($bits);
        $i = 0; $out = [];

        while ($i + 16 <= $n) {
            $type = ($bits[$i] << 1) | $bits[$i+1]; $i += 2;

            $val = 0;
            for ($k = 0; $k < 14; $k++) {
                $val = ($val << 1) | $bits[$i+$k];
            }
            $i += 14;

            if ($type === 0b00) { // temperature signed Q1
                if ($val & (1<<13)) $val -= (1<<14);
                $out['temperature'] = $val / 10.0;
            } elseif ($type === 0b01) { // humidity unsigned Q1
                $out['humidity'] = $val / 10.0;
            }
        }
        return $out;
    }

    public static function decodeBase64(string $b64): array
    {
        $raw = base64_decode($b64, true);
        return $raw === false ? [] : self::decode($raw);
    }

    private static function bytesToBits(string $raw): array
    {
        $bits = [];
        foreach (str_split($raw) as $ch) {
            $byte = ord($ch);
            for ($b = 7; $b >= 0; $b--) $bits[] = ($byte >> $b) & 1;
        }
        return $bits;
    }
}
