<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\TtnUplinkRequest;
use App\Models\Ruche;
use App\Models\Capteur;
use App\Models\Mesure;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Carbon\Carbon;

class TtnController extends Controller
{
    public function uplink(TtnUplinkRequest $request): JsonResponse
    {
        $data = $request->validated();
        $raw  = $request->all();

        $provided = $request->header('X-Webhook-Secret') ?: $request->query('secret');
        $expected = config('services.ttn.webhook_secret') ?? env('TTN_WEBHOOK_SECRET');
        if (empty($expected) || !hash_equals((string)$expected, (string)$provided)) {
            Log::warning('TTN: secret invalide ou manquant', [
                'ip' => $request->ip(),
                'ua' => $request->userAgent(),
            ]);
            return response()->json(['status' => 'forbidden'], 403);
        }

        $devEui   = Arr::get($data, 'end_device_ids.dev_eui');
        $deviceId = Arr::get($data, 'end_device_ids.device_id');

        $ts = Arr::get($data, 'received_at') ?? Arr::get($data, 'uplink_message.received_at');
        $measureAt = $ts ? Carbon::parse($ts) : now();

        $decoded  = Arr::get($data, 'uplink_message.decoded_payload', []);
        $frmB64   = Arr::get($data, 'uplink_message.frm_payload');

        $reference = $deviceId ?: $devEui;
        if (!$reference) {
            Log::warning('TTN uplink sans device_id/dev_eui', ['payload' => $raw]);
            return response()->json(['status' => 'ignored', 'reason' => 'no device identifier'], 202);
        }

        $ruche = Ruche::where('reference', $reference)->first();
        if (!$ruche) {
            Log::info('TTN uplink ruche inconnue', ['reference' => $reference]);
            return response()->json(['status' => 'ignored', 'reason' => 'unknown ruche reference', 'reference' => $reference], 202);
        }

        $created = [];
        $units = [
            'temperature' => '°C',
            'humidity'    => '%',
            'weight'      => 'kg',
            'battery'     => 'V',
        ];

        if (is_array($decoded) && !empty($decoded)) {
            foreach ($decoded as $key => $value) {
                if (is_array($value) || is_object($value)) {
                    continue;
                }
                $type = \Illuminate\Support\Str::slug($key, '_');

                $capteur = \App\Models\Capteur::firstOrCreate(
                    ['ruche_id' => $ruche->id, 'type' => $type],
                    ['unite' => $units[$type] ?? null, 'description' => 'auto via TTN']
                );

                $mesure = \App\Models\Mesure::create([
                    'capteur_id'     => $capteur->id,
                    'valeur_numeric' => is_numeric($value) ? (float)$value : null,
                    'valeur_text'    => is_numeric($value) ? null : (string)$value,
                    'mesure_at'      => $measureAt,
                    'raw_payload'    => $decoded,
                ]);
                $created[] = ['type' => $type, 'mesure_id' => $mesure->id];
            }
        } else {
            $capteur = \App\Models\Capteur::firstOrCreate(
                ['ruche_id' => $ruche->id, 'type' => 'raw'],
                ['description' => 'paquet TTN brut']
            );

            $mesure = \App\Models\Mesure::create([
                'capteur_id'     => $capteur->id,
                'valeur_numeric' => null,
                'valeur_text'    => $frmB64 ? ('b64:' . $frmB64) : null,
                'mesure_at'      => $measureAt,
                'raw_payload'    => $raw,
            ]);
            $created[] = ['type' => 'raw', 'mesure_id' => $mesure->id];
        }

        return response()->json([
            'status' => 'ok',
            'ruche_id' => $ruche->id,
            'reference' => $ruche->reference,
            'created' => $created,
        ]);
    }
}
