<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Carbon;

class TtnController extends Controller
{
    public function uplink(Request $request)
    {
        // --- Sécurité optionnelle via X-TTN-Secret ---
        $secret = env('TTN_SECRET');
        if ($secret) {
            $provided = $request->header('X-TTN-Secret');
            if (!$provided || !hash_equals($secret, $provided)) {
                return response()->json(['status' => 'unauthorized'], 401);
            }
        }

        try {
            DB::beginTransaction();

            $devId    = $request->input('end_device_ids.device_id', 'ruche-001-node');
            $rucheRef = (preg_match('/(ruche-\d+)/', $devId, $m) ? $m[1] : 'ruche-001');

            // Horodatage de MESURE (priorité à received_at ; fallback now)
            $received = $request->input('received_at');
            $at       = $received ? Carbon::parse($received) : now();

            // Trouver la ruche + capteurs existants (types => id)
            $ruche = DB::table('ruches')->where('reference', $rucheRef)->first();
            if (!$ruche) {
                DB::rollBack();
                return response()->json(['status'=>'not_found','message'=>"Ruche $rucheRef absente"], 404);
            }
            $capteurs = DB::table('capteurs')->where('ruche_id', $ruche->id)->pluck('id','type');

            // 1) decoded_payload (si Payload Formatter TTN)
            $decoded = $request->input('uplink_message.decoded_payload');
            $metrics = [];

            if (is_array($decoded) && (array_key_exists('temperature', $decoded) || array_key_exists('humidity', $decoded))) {
                if (isset($decoded['temperature'])) $metrics['temperature'] = (float)$decoded['temperature'];
                if (isset($decoded['humidity']))    $metrics['humidity']    = (float)$decoded['humidity'];
            } else {
                // 2) frm_payload base64 (notre binaire v1)
                $frm = $request->input('uplink_message.frm_payload');
                if (!$frm) return response()->json(['status'=>'bad_request','message'=>'no payload'], 400);
                $raw = base64_decode($frm, true);
                if ($raw === false || strlen($raw) === 0)
                    return response()->json(['status'=>'bad_request','message'=>'invalid base64'], 400);

                $metrics = $this->decodeBinaryV1($raw);
                if (empty($metrics))
                    return response()->json(['status'=>'bad_request','message'=>'no metrics decoded'], 400);
            }

            // Upsert pour chaque mesure disponible (évite 1062)
            $inserted = [];
            foreach (['temperature','humidity'] as $type) {
                if (!array_key_exists($type, $metrics)) continue;
                if (!isset($capteurs[$type])) continue;

                DB::table('mesures')->updateOrInsert(
                    ['capteur_id' => $capteurs[$type], 'measured_at' => $at],
                    ['valeur' => $metrics[$type], 'updated_at' => now(), 'created_at' => now()]
                );
                $inserted[] = ['type'=>$type, 'value'=>$metrics[$type], 'at'=>(string)$at];
            }

            DB::commit();
            return response()->json(['status'=>'ok','ruche'=>$rucheRef,'inserted'=>$inserted]);
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json(['status'=>'error','message'=>$e->getMessage()], 500);
        }
    }

    // v1: segments 16 bits = [type:2b][val:14b(Q1)], 00=temp(signed), 01=hum(unsigned)
    private function decodeBinaryV1(string $raw): array
    {
        $bits = $this->bytesToBits($raw);
        $n = count($bits); $i = 0; $out = [];
        while ($i + 16 <= $n) {
            $type = ($bits[$i] << 1) | $bits[$i+1]; $i += 2;
            $val = 0; for ($k=0; $k<14; $k++) { $val = ($val<<1) | $bits[$i+$k]; } $i += 14;

            if ($type === 0b00) { // temperature signed Q1
                if ($val & (1<<13)) $val -= (1<<14);
                $out['temperature'] = $val / 10.0;
            } elseif ($type === 0b01) { // humidity unsigned Q1
                $out['humidity'] = $val / 10.0;
            }
        }
        return $out;
    }

    private function bytesToBits(string $raw): array
    {
        $bits=[]; foreach (str_split($raw) as $ch) {
            $byte = ord($ch);
            for ($b=7; $b>=0; $b--) $bits[] = ($byte>>$b) & 1;
        }
        return $bits;
    }
}
