<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration {
    public function up(): void
    {
        if (!Schema::hasTable('alert_settings')) return;

        // 1) Trouver le vrai nom de la contrainte FOREIGN KEY sur alert_settings.ruche_id
        $fkNames = DB::select("
            SELECT tc.CONSTRAINT_NAME
            FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS tc
            JOIN INFORMATION_SCHEMA.KEY_COLUMN_USAGE kcu
              ON tc.CONSTRAINT_NAME = kcu.CONSTRAINT_NAME
             AND tc.TABLE_SCHEMA = kcu.TABLE_SCHEMA
             AND tc.TABLE_NAME   = kcu.TABLE_NAME
            WHERE tc.TABLE_SCHEMA = DATABASE()
              AND tc.TABLE_NAME   = 'alert_settings'
              AND tc.CONSTRAINT_TYPE = 'FOREIGN KEY'
              AND kcu.COLUMN_NAME = 'ruche_id'
        ");

        foreach ($fkNames as $row) {
            $name = $row->CONSTRAINT_NAME;
            try {
                DB::statement("ALTER TABLE `alert_settings` DROP FOREIGN KEY `{$name}`");
            } catch (\Throwable $e) {
                // ignore si déjà absent
            }
        }

        // 2) Supprimer tout index UNIQUE qui ne porte que sur ruche_id (nom variable)
        //    On détecte via SHOW INDEX puis on drop les Key_name concernés.
        $indexes = DB::select("SHOW INDEX FROM `alert_settings`");
        $uniqueSingleOnRuche = [];
        $colsByKey = [];

        foreach ($indexes as $idx) {
            $key = $idx->Key_name;
            $colsByKey[$key] ??= ['non_unique' => (int)$idx->Non_unique, 'cols' => []];
            $colsByKey[$key]['cols'][(int)$idx->Seq_in_index] = $idx->Column_name;
        }

        foreach ($colsByKey as $key => $meta) {
            if ($meta['non_unique'] === 0) { // unique
                ksort($meta['cols']);
                $cols = array_values($meta['cols']);
                if (count($cols) === 1 && $cols[0] === 'ruche_id') {
                    $uniqueSingleOnRuche[] = $key;
                }
            }
        }

        foreach ($uniqueSingleOnRuche as $key) {
            try {
                DB::statement("ALTER TABLE `alert_settings` DROP INDEX `{$key}`");
            } catch (\Throwable $e) {
                // ignore
            }
        }

        // 3) S'assurer d'un index simple (non unique) sur ruche_id (utile pour la FK)
        try {
            Schema::table('alert_settings', function (Blueprint $table) {
                $table->index('ruche_id', 'alert_settings_ruche_id_index');
            });
        } catch (\Throwable $e) {
            // ignore si déjà là
        }

        // 4) Créer l'unique composite (ruche_id, type)
        try {
            Schema::table('alert_settings', function (Blueprint $table) {
                $table->unique(['ruche_id','type'], 'alert_settings_ruche_type_unique');
            });
        } catch (\Throwable $e) {
            // ignore si déjà là
        }

        // 5) Recréer la FK ruche_id -> ruches(id)
        try {
            Schema::table('alert_settings', function (Blueprint $table) {
                $table->foreign('ruche_id')
                      ->references('id')->on('ruches')
                      ->cascadeOnDelete();
            });
        } catch (\Throwable $e) {
            // ignore si déjà là
        }
    }

    public function down(): void
    {
        if (!Schema::hasTable('alert_settings')) return;

        // Drop la FK (quel que soit le nom)
        $fkNames = DB::select("
            SELECT tc.CONSTRAINT_NAME
            FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS tc
            JOIN INFORMATION_SCHEMA.KEY_COLUMN_USAGE kcu
              ON tc.CONSTRAINT_NAME = kcu.CONSTRAINT_NAME
             AND tc.TABLE_SCHEMA = kcu.TABLE_SCHEMA
             AND tc.TABLE_NAME   = kcu.TABLE_NAME
            WHERE tc.TABLE_SCHEMA = DATABASE()
              AND tc.TABLE_NAME   = 'alert_settings'
              AND tc.CONSTRAINT_TYPE = 'FOREIGN KEY'
              AND kcu.COLUMN_NAME = 'ruche_id'
        ");

        foreach ($fkNames as $row) {
            $name = $row->CONSTRAINT_NAME;
            try { DB::statement("ALTER TABLE `alert_settings` DROP FOREIGN KEY `{$name}`"); } catch (\Throwable $e) {}
        }

        // Drop l'unique composite
        try {
            Schema::table('alert_settings', function (Blueprint $table) {
                $table->dropUnique('alert_settings_ruche_type_unique');
            });
        } catch (\Throwable $e) {}

        // Drop l'index non-unique ruche_id
        try {
            Schema::table('alert_settings', function (Blueprint $table) {
                $table->dropIndex('alert_settings_ruche_id_index');
            });
        } catch (\Throwable $e) {}

        // Reposer un unique sur ruche_id (état antérieur)
        try {
            Schema::table('alert_settings', function (Blueprint $table) {
                $table->unique('ruche_id', 'alert_settings_ruche_id_unique');
            });
        } catch (\Throwable $e) {}

        // Recréer la FK
        try {
            Schema::table('alert_settings', function (Blueprint $table) {
                $table->foreign('ruche_id')
                      ->references('id')->on('ruches')
                      ->cascadeOnDelete();
            });
        } catch (\Throwable $e) {}
    }
};
