<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Carbon\Carbon;

class AddWeightBatterySeeder extends Seeder
{
    /**
     * Crée les capteurs manquants "weight" et "battery" pour chaque ruche,
     * et backfill des mesures sur 24 h (toutes les 10 minutes).
     * Compatible avec différents schémas de colonnes.
     */
    public function run(): void
    {
        $now  = Carbon::now()->second(0)->micro(0);
        $from = $now->copy()->subHours(24);

        // Déterminer la colonne "référence" existante dans `ruches`
        $rucheRefCol = $this->firstExistingColumn('ruches', ['ref', 'reference', 'slug', 'name', 'nom']);
        // On sélectionne au minimum l'id
        $cols = ['id'];
        if ($rucheRefCol) $cols[] = $rucheRefCol;

        $ruches = DB::table('ruches')->select($cols)->get();
        if ($ruches->isEmpty()) {
            $this->command?->warn('Aucune ruche trouvée. Seed des ruches d’abord.');
            return;
        }

        foreach ($ruches as $ruche) {
            $rucheLabel = $rucheRefCol ? ($ruche->{$rucheRefCol} ?? ('ruche-'.$ruche->id)) : ('ruche-'.$ruche->id);

            // S’assurer que les capteurs existent
            $weightId  = $this->ensureCapteur($ruche->id, 'weight',  'Balance');
            $batteryId = $this->ensureCapteur($ruche->id, 'battery', 'Batterie (V)');

            $this->command?->info("Ruche {$rucheLabel} (#{$ruche->id}) : weight={$weightId}, battery={$batteryId}");

            // Backfill 24 h (pas 10 minutes)
            $t = $from->copy();
            $weight  = 1.00;  // kg
            $battery = 4.10;  // V

            while ($t <= $now) {
                // petites variations
                $weight  += (mt_rand(-5, 5) / 1000.0);   // ±0.005 kg
                $battery += (mt_rand(-2, 1) / 1000.0);   // légère baisse

                // bornes
                $weight  = max(0.50, min($weight,  5.00));
                $battery = max(3.60, min($battery, 4.20));

                $this->insertIfMissing($weightId,  $t, round($weight,  3));
                $this->insertIfMissing($batteryId, $t, round($battery, 3));

                $t->addMinutes(10);
            }
        }
    }

    /**
     * Crée/retourne un capteur pour une ruche.
     * Remplit un label si une colonne correspondante existe (reference/name/nom...),
     * sinon n’insère que les champs obligatoires.
     */
    private function ensureCapteur(int $rucheId, string $type, string $labelFallback): int
    {
        $capteur = DB::table('capteurs')
            ->where('ruche_id', $rucheId)
            ->where('type', $type)
            ->first();

        if ($capteur) return (int) $capteur->id;

        // Colonnes optionnelles pour libellé
        $labelCol = $this->firstExistingColumn('capteurs', ['reference', 'label', 'name', 'nom']);

        $data = [
            'ruche_id'   => $rucheId,
            'type'       => $type,
            'created_at' => now(),
            'updated_at' => now(),
        ];
        if ($labelCol) {
            $data[$labelCol] = $labelFallback;
        }

        return (int) DB::table('capteurs')->insertGetId($data);
    }

    /**
     * Insert mesure si inexistante pour (capteur_id, measured_at).
     * Compatible avec schémas: valeur/text + value_numeric/decimal.
     */
    private function insertIfMissing(int $capteurId, Carbon $measuredAt, float $value): void
    {
        $ts = $measuredAt->toDateTimeString();

        $exists = DB::table('mesures')
            ->where('capteur_id', $capteurId)
            ->where('measured_at', $ts)
            ->exists();

        if ($exists) return;

        $data = [
            'capteur_id'    => $capteurId,
            'measured_at'   => $ts,
            'created_at'    => now(),
            'updated_at'    => now(),
        ];

        // colonnes valeur possibles
        $hasValeur = Schema::hasColumn('mesures', 'valeur');
        $hasValueN = Schema::hasColumn('mesures', 'value_numeric');

        if ($hasValeur) $data['valeur'] = (string) $value;
        if ($hasValueN) $data['value_numeric'] = $value;

        // Si aucune des deux colonnes n’existe, on tente quand même "value"
        if (!$hasValeur && !$hasValueN) {
            if (Schema::hasColumn('mesures', 'value')) {
                $data['value'] = $value;
            } else {
                // Dernier recours : on refuse d’insérer pour éviter une erreur SQL
                $this->command?->error("Table 'mesures' n’a ni 'valeur', ni 'value_numeric', ni 'value' — insertion ignorée.");
                return;
            }
        }

        DB::table('mesures')->insert($data);
    }

    /**
     * Renvoie la première colonne existante parmi une liste.
     */
    private function firstExistingColumn(string $table, array $candidates): ?string
    {
        foreach ($candidates as $c) {
            if (Schema::hasColumn($table, $c)) return $c;
        }
        return null;
    }
}
