@extends('layouts.app')

@section('title', 'Alertes — '.$ref)

@section('content')
<div class="max-w-7xl mx-auto space-y-6">

  {{-- En-tête --}}
  <div class="flex flex-col md:flex-row md:items-center md:justify-between gap-3">
    <div>
      <h1 class="text-2xl font-bold">Alertes — <span class="text-indigo-700">{{ $ref }}</span></h1>
      <p class="text-sm text-gray-500">Gérez les seuils par ruche (min/max, hystérésis, activation).</p>
    </div>

    {{-- Sélecteur de ruche + Bootstrap --}}
    <div class="flex flex-wrap items-center gap-2">
      <form method="GET" action="{{ route('alerts.index') }}" class="flex items-center gap-2">
        <select name="ref" class="border rounded-lg px-3 py-2" onchange="this.form.submit()">
          @foreach($ruches as $r)
            <option value="{{ $r }}" {{ $r === $ref ? 'selected' : '' }}>{{ $r }}</option>
          @endforeach
        </select>
        <noscript>
          <button class="px-3 py-2 rounded-lg border">Changer</button>
        </noscript>
      </form>

      <form method="POST" action="{{ route('alerts.bootstrap') }}">
        @csrf
        <input type="hidden" name="ref" value="{{ $ref }}">
        <button class="px-3 py-2 rounded-lg bg-emerald-600 text-white hover:bg-emerald-700">
          Initialiser les seuils manquants
        </button>
      </form>
    </div>
  </div>

  {{-- Flash message / erreurs --}}
  @if (session('status'))
    <div class="p-3 rounded-lg bg-emerald-50 text-emerald-800 border border-emerald-200">
      {{ session('status') }}
    </div>
  @endif

  @if ($errors->any())
    <div class="p-3 rounded-lg bg-red-50 text-red-800 border border-red-200">
      <div class="font-semibold mb-1">Erreurs :</div>
      <ul class="list-disc list-inside text-sm">
        @foreach ($errors->all() as $error)
          <li>{{ $error }}</li>
        @endforeach
      </ul>
    </div>
  @endif

  {{-- Tableau des réglages --}}
  <div class="bg-white rounded-2xl shadow p-4">
    <div class="flex items-center justify-between mb-3">
      <h3 class="font-semibold">Réglages des seuils</h3>
      <div class="text-xs text-gray-500">
        Astuce : laissez <em>vide</em> pour « sans seuil ».
      </div>
    </div>

    @if($settings->isEmpty())
      <div class="text-sm text-gray-500">
        Aucun réglage pour cette ruche. Cliquez sur <b>“Initialiser les seuils manquants”</b>.
      </div>
    @else
      <div class="overflow-x-auto">
        <table class="min-w-full text-sm">
          <thead>
            <tr class="text-left text-gray-600 border-b">
              <th class="px-3 py-2">Type</th>
              <th class="px-3 py-2">Seuil min</th>
              <th class="px-3 py-2">Seuil max</th>
              <th class="px-3 py-2">Hystérésis</th>
              <th class="px-3 py-2">Activé</th>
              <th class="px-3 py-2 text-right">Actions</th>
            </tr>
          </thead>
          <tbody>
            @foreach($settings as $s)
              <tr class="border-t align-top">
                <td class="px-3 py-2 font-medium capitalize">
                  {{ $s->type }}
                  <div class="text-xs text-gray-400">
                    @switch($s->type)
                      @case('temperature') °C @break
                      @case('humidity') % @break
                      @case('weight') kg @break
                      @case('battery') V @break
                      @default —
                    @endswitch
                  </div>
                </td>

                {{-- Formulaire d’édition ligne par ligne --}}
                <form method="POST" action="{{ route('alerts.update', $s->id) }}" class="contents alert-row-form" data-type="{{ $s->type }}">
                  @csrf
                  <td class="px-3 py-2">
                    <input type="number" step="0.01" name="threshold_min" class="w-36 border rounded-lg px-2 py-1"
                           value="{{ $s->threshold_min }}"
                           placeholder="(vide)">
                  </td>
                  <td class="px-3 py-2">
                    <input type="number" step="0.01" name="threshold_max" class="w-36 border rounded-lg px-2 py-1"
                           value="{{ $s->threshold_max }}"
                           placeholder="(vide)">
                  </td>
                  <td class="px-3 py-2">
                    <input type="number" step="0.01" name="hysteresis" class="w-28 border rounded-lg px-2 py-1"
                           value="{{ $s->hysteresis }}" required>
                    <div class="text-[11px] text-gray-400 mt-1">Bande anti-rebond</div>
                  </td>
                  <td class="px-3 py-2">
                    <input type="hidden" name="enabled" value="0">
                    <label class="inline-flex items-center gap-2">
                      <input type="checkbox" name="enabled" value="1" {{ $s->enabled ? 'checked' : '' }}>
                      <span class="text-xs">
                        <span class="inline-flex items-center px-2 py-0.5 rounded-full
                          {{ $s->enabled ? 'bg-emerald-50 text-emerald-700 border border-emerald-200' : 'bg-gray-100 text-gray-600 border border-gray-200' }}">
                          {{ $s->enabled ? 'Actif' : 'Off' }}
                        </span>
                      </span>
                    </label>
                  </td>
                  <td class="px-3 py-2">
                    <div class="flex items-center gap-2 justify-end">
                      <button class="px-3 py-2 rounded-lg bg-indigo-600 text-white hover:bg-indigo-700">
                        Enregistrer
                      </button>
                </form>

                      <form method="POST" action="{{ route('alerts.destroy', $s->id) }}"
                            onsubmit="return confirm('Supprimer ce réglage ?');">
                        @csrf
                        @method('DELETE')
                        <button class="px-3 py-2 rounded-lg bg-red-50 text-red-700 hover:bg-red-100 border border-red-200">
                          Supprimer
                        </button>
                      </form>
                    </div>

                    <div class="text-[11px] text-gray-400 mt-2 text-right">
                      Si les deux seuils sont renseignés, <b>min ≤ max</b>.
                    </div>
                  </td>
              </tr>
            @endforeach
          </tbody>
        </table>
      </div>
    @endif
  </div>

  {{-- Dernières alertes déclenchées (lecture seule) --}}
  <div class="bg-white rounded-2xl shadow p-4">
    <div class="flex items-center justify-between mb-3">
      <h3 class="font-semibold">Dernières alertes</h3>
    </div>

    @if($latestAlerts->isEmpty())
      <div class="text-sm text-gray-500">Aucune alerte enregistrée pour l’instant.</div>
    @else
      <div class="overflow-x-auto">
        <table class="min-w-full text-sm">
          <thead>
            <tr class="text-left text-gray-600 border-b">
              <th class="px-3 py-2">Date</th>
              <th class="px-3 py-2">Type</th>
              <th class="px-3 py-2">Valeur</th>
              <th class="px-3 py-2">Seuil franchi</th>
              <th class="px-3 py-2">Message</th>
            </tr>
          </thead>
          <tbody>
            @foreach($latestAlerts as $a)
              <tr class="border-t">
                <td class="px-3 py-2">{{ data_get($a,'created_at','—') }}</td>
                <td class="px-3 py-2 capitalize">{{ data_get($a,'type','—') }}</td>
                <td class="px-3 py-2">{{ data_get($a,'value','—') }}</td>
                {{-- Cellule robuste sur la direction --}}
                <td class="px-3 py-2">
                  @php $dir = data_get($a, 'direction'); @endphp
                  @if ($dir === 'up')
                    <span class="text-red-600 font-medium">MAX</span>
                  @elseif ($dir === 'down')
                    <span class="text-blue-600 font-medium">MIN</span>
                  @elseif (!is_null($dir))
                    <span class="text-gray-600">{{ $dir }}</span>
                  @else
                    —
                  @endif
                </td>
                <td class="px-3 py-2">{{ data_get($a,'message','—') }}</td>
              </tr>
            @endforeach
          </tbody>
        </table>
      </div>
    @endif
  </div>

</div>
@endsection

@push('scripts')
<script>
/**
 * Validation front douce :
 * - Autorise les champs vides (=> sans seuil)
 * - Si min & max sont tous deux renseignés : impose min <= max
 */
(function(){
  document.querySelectorAll('.alert-row-form').forEach(form => {
    form.addEventListener('submit', (e) => {
      const minI = form.querySelector('input[name="threshold_min"]');
      const maxI = form.querySelector('input[name="threshold_max"]');
      const hysI = form.querySelector('input[name="hysteresis"]');

      const minV = (minI?.value ?? '').trim();
      const maxV = (maxI?.value ?? '').trim();

      [minI, maxI, hysI].forEach(i => i?.classList.remove('ring-2','ring-red-300'));

      if (!hysI?.value?.trim()) {
        hysI.classList.add('ring-2','ring-red-300');
        hysI.focus();
        e.preventDefault();
        return false;
      }

      if (minV !== '' && maxV !== '') {
        const minN = Number(minV), maxN = Number(maxV);
        if (!Number.isFinite(minN) || !Number.isFinite(maxN) || minN > maxN) {
          minI.classList.add('ring-2','ring-red-300');
          maxI.classList.add('ring-2','ring-red-300');
          alert('Veuillez vérifier les seuils : si min et max sont tous deux renseignés, min doit être ≤ max.');
          e.preventDefault();
          return false;
        }
      }
      return true;
    });
  });
})();
</script>
@endpush
